<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @package    SiteEngineAI_Chat
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The public-facing functionality of the plugin.
 */
class SiteEngineAI_Chat_Public {

	/**
	 * Convert lowercase keys to camelCase (migration helper).
	 *
	 * @param array $styles Styles array with potentially lowercase keys.
	 * @return array Styles array with camelCase keys.
	 */
	private function convert_styles_keys_to_camelcase( $styles ) {
		if ( ! is_array( $styles ) || empty( $styles ) ) {
			return $styles;
		}

		// Get key mapping from config
		$key_mapping = SiteEngineAI_Chat_Config::get_key_mapping();

		$converted = array();

		foreach ( $styles as $key => $value ) {
			// Check if key is lowercase and needs conversion
			if ( isset( $key_mapping[ strtolower( $key ) ] ) && $key !== $key_mapping[ strtolower( $key ) ] ) {
				$new_key = $key_mapping[ strtolower( $key ) ];
				$converted[ $new_key ] = $value;
			} else {
				$converted[ $key ] = $value;
			}
		}

		return $converted;
	}

	/**
	 * Enqueue chat script.
	 */
	public function enqueue_scripts() {
		// Check if chat is enabled
		$default_options = SiteEngineAI_Chat_Config::get_default_options();
		if ( ! get_option( SiteEngineAI_Chat_Config::OPTION_ENABLED, $default_options[ SiteEngineAI_Chat_Config::OPTION_ENABLED ] ) ) {
			return;
		}

		$use_local_styles = get_option( SiteEngineAI_Chat_Config::OPTION_USE_LOCAL_STYLES, $default_options[ SiteEngineAI_Chat_Config::OPTION_USE_LOCAL_STYLES ] );
		$local_styles     = get_option( SiteEngineAI_Chat_Config::OPTION_STYLES, $default_options[ SiteEngineAI_Chat_Config::OPTION_STYLES ] );
		
		// Convert lowercase keys to camelCase if needed (migration)
		$local_styles = $this->convert_styles_keys_to_camelcase( $local_styles );

		// Register and enqueue main chat script
		wp_enqueue_script(
			'siteengineai-chat',
			SiteEngineAI_Chat_Config::get_chat_script_url(),
			array(),
			SiteEngineAI_Chat_Config::PLUGIN_VERSION,
			true
		);

		// If local styles are enabled, pass them to chat.js via inline script
		if ( $use_local_styles && ! empty( $local_styles ) ) {
			// wp_json_encode properly escapes JSON for JavaScript context.
			$js_styles = wp_json_encode( $local_styles, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES );
			$inline_script = 'window.seChatLocalStyles = ' . $js_styles . ';';
			wp_add_inline_script( 'siteengineai-chat', $inline_script, 'before' );
		}
	}

	/**
	 * Add chat script to footer (deprecated, kept for backward compatibility).
	 * 
	 * @deprecated Use enqueue_scripts() instead.
	 */
	public function add_chat_script() {
		// This method is kept for backward compatibility but should not be used.
		// Scripts are now enqueued via enqueue_scripts() method.
	}
}

