<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package    SiteEngineAI_Chat
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The admin-specific functionality of the plugin.
 */
class SiteEngineAI_Chat_Admin {

	/**
	 * Site checker instance.
	 *
	 * @var SiteEngineAI_Chat_Site_Checker
	 */
	private $site_checker;

	/**
	 * Constructor.
	 */
	public function __construct() {
		require_once SE_CHAT_PLUGIN_DIR . 'includes/class-siteengineai-site-checker.php';
		$this->site_checker = new SiteEngineAI_Chat_Site_Checker();
	}

	/**
	 * Register the stylesheets for the admin area.
	 */
	public function enqueue_styles() {
		$screen = get_current_screen();
		if ( SiteEngineAI_Chat_Config::ADMIN_SCREEN_ID !== $screen->id ) {
			return;
		}

		wp_enqueue_style(
			'siteengineai-chat-admin',
			SE_CHAT_PLUGIN_URL . 'admin/css/admin.css',
			array(),
			SE_CHAT_VERSION,
			'all'
		);
	}

	/**
	 * Register the JavaScript for the admin area.
	 */
	public function enqueue_scripts() {
		$screen = get_current_screen();
		if ( SiteEngineAI_Chat_Config::ADMIN_SCREEN_ID !== $screen->id ) {
			return;
		}

		wp_enqueue_script( 'wp-color-picker' );
		wp_enqueue_style( 'wp-color-picker' );

		wp_enqueue_script(
			'siteengineai-chat-admin',
			SE_CHAT_PLUGIN_URL . 'admin/js/admin.js',
			array( 'jquery', 'wp-color-picker' ),
			SE_CHAT_VERSION,
			false
		);

		// Localize script for AJAX
		wp_localize_script(
			'siteengineai-chat-admin',
			'seChatAdmin',
			array(
				'ajaxurl'      => admin_url( 'admin-ajax.php' ),
				'nonce'        => wp_create_nonce( SiteEngineAI_Chat_Config::NONCE_NOTICE ),
				'ajax_action'  => SiteEngineAI_Chat_Config::AJAX_ACTION_NOTICE,
			)
		);
	}

	/**
	 * Register the administration menu for this plugin.
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'SiteEngineAI Chat Settings', 'siteengineai-chat' ),
			__( 'SiteEngineAI Chat', 'siteengineai-chat' ),
			'manage_options',
			SiteEngineAI_Chat_Config::ADMIN_PAGE_SLUG,
			array( $this, 'display_plugin_setup_page' ),
			'dashicons-format-chat',
			30
		);
	}

	/**
	 * Register settings.
	 */
	public function register_settings() {
		$default_options = SiteEngineAI_Chat_Config::get_default_options();
		
		register_setting( SiteEngineAI_Chat_Config::NONCE_SETTINGS, SiteEngineAI_Chat_Config::OPTION_ENABLED, array( 
			'type'              => 'boolean',
			'sanitize_callback' => array( $this, 'sanitize_boolean' ),
			'default'           => $default_options[ SiteEngineAI_Chat_Config::OPTION_ENABLED ],
		) );
		register_setting( SiteEngineAI_Chat_Config::NONCE_SETTINGS, SiteEngineAI_Chat_Config::OPTION_USE_LOCAL_STYLES, array( 
			'type'              => 'boolean',
			'sanitize_callback' => array( $this, 'sanitize_boolean' ),
			'default'           => $default_options[ SiteEngineAI_Chat_Config::OPTION_USE_LOCAL_STYLES ],
		) );
		register_setting( SiteEngineAI_Chat_Config::NONCE_SETTINGS, SiteEngineAI_Chat_Config::OPTION_STYLES, array( 
			'type'              => 'array',
			'sanitize_callback' => array( $this, 'sanitize_styles' ),
			'default'           => $default_options[ SiteEngineAI_Chat_Config::OPTION_STYLES ],
		) );
	}

	/**
	 * Sanitize boolean value.
	 */
	public function sanitize_boolean( $value ) {
		return filter_var( $value, FILTER_VALIDATE_BOOLEAN );
	}

	/**
	 * Convert lowercase keys to camelCase (migration helper).
	 *
	 * @param array $styles Styles array with potentially lowercase keys.
	 * @return array Styles array with camelCase keys.
	 */
	private function convert_styles_keys_to_camelcase( $styles ) {
		if ( ! is_array( $styles ) || empty( $styles ) ) {
			return $styles;
		}

		// Get key mapping from config
		$key_mapping = SiteEngineAI_Chat_Config::get_key_mapping();

		$converted = array();
		$needs_update = false;

		foreach ( $styles as $key => $value ) {
			// Check if key is lowercase and needs conversion
			if ( isset( $key_mapping[ strtolower( $key ) ] ) && $key !== $key_mapping[ strtolower( $key ) ] ) {
				$new_key = $key_mapping[ strtolower( $key ) ];
				$converted[ $new_key ] = $value;
				$needs_update = true;
			} else {
				$converted[ $key ] = $value;
			}
		}

		// If conversion happened, save the converted array
		if ( $needs_update ) {
			update_option( SiteEngineAI_Chat_Config::OPTION_STYLES, $converted );
		}

		return $converted;
	}

	/**
	 * Sanitize styles array.
	 */
	public function sanitize_styles( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		// Get allowed style keys from config
		$allowed_keys = SiteEngineAI_Chat_Config::get_allowed_style_keys();

		$sanitized = array();
		foreach ( $input as $key => $value ) {
			// Only save if key is in allowed list (preserve camelCase)
			if ( in_array( $key, $allowed_keys, true ) ) {
				$sanitized[ $key ] = sanitize_text_field( $value );
			}
		}

		return $sanitized;
	}

	/**
	 * Show admin notices.
	 */
	public function show_admin_notices() {
		// Only show on admin pages
		if ( ! is_admin() ) {
			return;
		}

		// Don't show on plugin settings page (it has its own notice)
		$screen = get_current_screen();
		if ( $screen && SiteEngineAI_Chat_Config::ADMIN_SCREEN_ID === $screen->id ) {
			return;
		}

		// Check if user dismissed the notice
		$user_dismissed = get_user_meta( get_current_user_id(), SiteEngineAI_Chat_Config::USER_META_NOTICE_DISMISSED, true );
		if ( $user_dismissed && ( time() - $user_dismissed ) < 86400 ) { // 24 hours
			return;
		}

		// Only show if chat is enabled
		$default_options = SiteEngineAI_Chat_Config::get_default_options();
		if ( ! get_option( SiteEngineAI_Chat_Config::OPTION_ENABLED, $default_options[ SiteEngineAI_Chat_Config::OPTION_ENABLED ] ) ) {
			return;
		}

		// Check site registration
		$check_result = $this->site_checker->check_site_registration();

		if ( ! $check_result['registered'] ) {
			$site_url = $this->site_checker->get_site_url();
			$register_url = SiteEngineAI_Chat_Config::get_registration_url();
			?>
			<div class="notice notice-warning is-dismissible se-chat-site-notice">
				<p>
					<strong><?php esc_html_e( 'SiteEngineAI Chat:', 'siteengineai-chat' ); ?></strong>
					<?php
					printf(
						/* translators: 1: Site URL, 2: Register URL */
						esc_html__( 'Your site %1$s is not registered in SiteEngineAI. Please %2$s to register your site and enable the chat functionality.', 'siteengineai-chat' ),
						'<code>' . esc_html( $site_url ) . '</code>',
						'<a href="' . esc_url( $register_url ) . '" target="_blank" rel="noopener noreferrer">' . esc_html__( 'visit SiteEngineAI.com', 'siteengineai-chat' ) . '</a>'
					);
					?>
				</p>
				<p>
					<a href="<?php echo esc_url( $register_url ); ?>" target="_blank" rel="noopener noreferrer" class="button button-primary">
						<?php esc_html_e( 'Register Site on SiteEngineAI', 'siteengineai-chat' ); ?>
					</a>
					<button type="button" class="button se-chat-dismiss-notice" data-action="dismiss">
						<?php esc_html_e( 'Dismiss', 'siteengineai-chat' ); ?>
					</button>
					<button type="button" class="button se-chat-check-again" data-action="check">
						<?php esc_html_e( 'Check Again', 'siteengineai-chat' ); ?>
					</button>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Handle AJAX requests for notice actions.
	 */
	public function handle_ajax_requests() {
		// Check nonce
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), SiteEngineAI_Chat_Config::NONCE_NOTICE ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce', 'siteengineai-chat' ) ) );
		}

		// Check permissions
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'siteengineai-chat' ) ) );
		}

		$action = isset( $_POST['action_type'] ) ? sanitize_text_field( wp_unslash( $_POST['action_type'] ) ) : '';

		switch ( $action ) {
			case 'dismiss':
				// Dismiss notice for current user
				update_user_meta( get_current_user_id(), SiteEngineAI_Chat_Config::USER_META_NOTICE_DISMISSED, time() );
				wp_send_json_success( array( 'message' => __( 'Notice dismissed', 'siteengineai-chat' ) ) );
				break;

			case 'check':
				// Clear cache and check again
				$this->site_checker->clear_cache();
				$result = $this->site_checker->check_site_registration();
				wp_send_json_success(
					array(
						'registered' => $result['registered'],
						'message'    => $result['message'],
					)
				);
				break;

			default:
				wp_send_json_error( array( 'message' => __( 'Invalid action', 'siteengineai-chat' ) ) );
		}
	}

	/**
	 * Render the settings page.
	 */
	public function display_plugin_setup_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'siteengineai-chat' ) );
		}

		// Handle form submission FIRST
		$settings_saved = false;
		$default_options = SiteEngineAI_Chat_Config::get_default_options();
		
		if ( isset( $_POST['se_chat_save'] ) ) {
			// Verify nonce
			if ( ! check_admin_referer( SiteEngineAI_Chat_Config::NONCE_SETTINGS, 'se_chat_nonce' ) ) {
				wp_die( esc_html__( 'Security check failed. Please try again.', 'siteengineai-chat' ) );
			}
			// Save enabled option
			$enabled_value = isset( $_POST['se_chat_enabled'] ) && '1' === $_POST['se_chat_enabled'] ? true : false;
			update_option( SiteEngineAI_Chat_Config::OPTION_ENABLED, $enabled_value );
			
			// Save use local styles option
			$use_local_styles_value = isset( $_POST['se_chat_use_local_styles'] ) && '1' === $_POST['se_chat_use_local_styles'] ? true : false;
			update_option( SiteEngineAI_Chat_Config::OPTION_USE_LOCAL_STYLES, $use_local_styles_value );

			// Save styles
			if ( isset( $_POST['se_chat_styles'] ) && is_array( $_POST['se_chat_styles'] ) ) {
				// Get current styles to preserve values not in form
				$current_styles = get_option( SiteEngineAI_Chat_Config::OPTION_STYLES, $default_options[ SiteEngineAI_Chat_Config::OPTION_STYLES ] );

				// Get default styles from config
				$default_styles = SiteEngineAI_Chat_Config::get_default_styles();
				
				// Start with defaults
				$styles_to_save = $default_styles;
				
				// Merge with current styles to preserve existing values
				$styles_to_save = wp_parse_args( $current_styles, $styles_to_save );
				
				// Get allowed keys from config
				$allowed_keys = SiteEngineAI_Chat_Config::get_allowed_style_keys();
				
				// Unslash and sanitize POST data
				$post_styles = wp_unslash( $_POST['se_chat_styles'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				
				foreach ( $post_styles as $key => $value ) {
					if ( in_array( $key, $allowed_keys, true ) ) {
						$sanitized_value = sanitize_text_field( $value );
						$styles_to_save[ $key ] = $sanitized_value;
					}
				}

				// Save styles
				update_option( SiteEngineAI_Chat_Config::OPTION_STYLES, $styles_to_save );
			}

			$settings_saved = true;
		}

		// Check site registration for settings page
		$check_result = $this->site_checker->check_site_registration();
		$site_checker = $this->site_checker;

		// Get current options
		$enabled          = get_option( SiteEngineAI_Chat_Config::OPTION_ENABLED, $default_options[ SiteEngineAI_Chat_Config::OPTION_ENABLED ] );
		$use_local_styles = get_option( SiteEngineAI_Chat_Config::OPTION_USE_LOCAL_STYLES, $default_options[ SiteEngineAI_Chat_Config::OPTION_USE_LOCAL_STYLES ] );
		$styles           = get_option( SiteEngineAI_Chat_Config::OPTION_STYLES, $default_options[ SiteEngineAI_Chat_Config::OPTION_STYLES ] );
		
		// Convert old lowercase keys to camelCase (migration)
		$styles = $this->convert_styles_keys_to_camelcase( $styles );

		// Pass variables to the view
		include_once SE_CHAT_PLUGIN_DIR . 'admin/views/settings-page.php';
	}
}

