<?php
/**
 * Check if site is registered in SiteEngineAI.
 *
 * @package    SiteEngineAI_Chat
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Site registration checker.
 */
class SiteEngineAI_Chat_Site_Checker {


	/**
	 * Check if current site is registered in SiteEngineAI.
	 *
	 * @return array {
	 *     @type bool   $registered Whether site is registered.
	 *     @type string $message    Error message if not registered.
	 * }
	 */
	public function check_site_registration() {
		// Get site URL
		$site_url = home_url();

		// Check cache first
		$cached = get_transient( SiteEngineAI_Chat_Config::CACHE_KEY_SITE_REGISTERED );
		if ( false !== $cached ) {
			return $cached;
		}

		// Make API request
		$api_url = add_query_arg(
			array(
				'site_url' => $site_url,
				'version'  => 0,
			),
			SiteEngineAI_Chat_Config::get_api_config_url()
		);

		$response = wp_remote_get(
			$api_url,
			array(
				'timeout'     => 10,
				'sslverify'   => true,
				'headers'     => array(
					'Origin' => $site_url,
				),
			)
		);

		// Check for errors
		if ( is_wp_error( $response ) ) {
			$result = array(
				'registered' => false,
				'message'    => __( 'Unable to check site registration. Please try again later.', 'siteengineai-chat' ),
				'error'       => $response->get_error_message(),
			);
		} else {
			$status_code = wp_remote_retrieve_response_code( $response );

			if ( 404 === $status_code ) {
				// Site not registered
				$result = array(
					'registered' => false,
					'message'    => __( 'This site is not registered in SiteEngineAI.', 'siteengineai-chat' ),
				);
			} elseif ( 200 === $status_code || 304 === $status_code ) {
				// Site is registered
				$result = array(
					'registered' => true,
					'message'    => '',
				);
			} else {
				// Other error
				$result = array(
					'registered' => false,
					'message'    => __( 'Unable to verify site registration.', 'siteengineai-chat' ),
					// translators: %d is the HTTP status code.
					'error'      => sprintf( __( 'HTTP %d', 'siteengineai-chat' ), $status_code ),
				);
			}
		}

		// Cache result
		set_transient( SiteEngineAI_Chat_Config::CACHE_KEY_SITE_REGISTERED, $result, SiteEngineAI_Chat_Config::CACHE_EXPIRATION );

		return $result;
	}

	/**
	 * Clear cache.
	 */
	public function clear_cache() {
		delete_transient( SiteEngineAI_Chat_Config::CACHE_KEY_SITE_REGISTERED );
	}

	/**
	 * Get site URL for display.
	 *
	 * @return string
	 */
	public function get_site_url() {
		return home_url();
	}
}

